#!/usr/bin/env python3
"""
CloudGuard Portfolio: AWS Account Creation Automation Script

This script automates the creation of member accounts in your AWS Organization.
It uses the AWS SDK for Python (boto3) to create accounts and move them to the correct OU.

Prerequisites:
- AWS CLI configured with credentials for the management account
- boto3 installed: pip install boto3
- Appropriate IAM permissions to create accounts and manage Organizations

Usage:
    python3 create_aws_account.py --name cloudguard-workloads-ml-01 --email your.email+cloudguard-workloads-ml-01@gmail.com --ou Workloads
"""

import boto3
import argparse
import time
import sys

def get_ou_id(org_client, ou_name):
    """
    Get the OU ID by name.
    
    Args:
        org_client: boto3 Organizations client
        ou_name: Name of the OU (e.g., 'Workloads')
    
    Returns:
        OU ID string or None if not found
    """
    try:
        # Get the root ID first
        roots = org_client.list_roots()
        root_id = roots['Roots'][0]['Id']
        
        # List all OUs under the root
        ous = org_client.list_organizational_units_for_parent(ParentId=root_id)
        
        for ou in ous['OrganizationalUnits']:
            if ou['Name'] == ou_name:
                return ou['Id']
        
        print(f"ERROR: OU '{ou_name}' not found.")
        return None
    except Exception as e:
        print(f"ERROR getting OU ID: {e}")
        return None

def create_account(org_client, account_name, email):
    """
    Create a new AWS account in the organization.
    
    Args:
        org_client: boto3 Organizations client
        account_name: Name for the new account
        email: Email address for the account root user
    
    Returns:
        Account creation request ID or None if failed
    """
    try:
        print(f"Creating account: {account_name}")
        print(f"Email: {email}")
        
        response = org_client.create_account(
            Email=email,
            AccountName=account_name,
            RoleName='OrganizationAccountAccessRole',
            IamUserAccessToBilling='ALLOW'
        )
        
        request_id = response['CreateAccountStatus']['Id']
        print(f"Account creation initiated. Request ID: {request_id}")
        return request_id
    except Exception as e:
        print(f"ERROR creating account: {e}")
        return None

def wait_for_account_creation(org_client, request_id, timeout=300):
    """
    Wait for account creation to complete.
    
    Args:
        org_client: boto3 Organizations client
        request_id: Account creation request ID
        timeout: Maximum time to wait in seconds (default: 300)
    
    Returns:
        Account ID if successful, None otherwise
    """
    print("Waiting for account creation to complete...")
    start_time = time.time()
    
    while True:
        if time.time() - start_time > timeout:
            print("ERROR: Account creation timed out.")
            return None
        
        try:
            response = org_client.describe_create_account_status(
                CreateAccountRequestId=request_id
            )
            
            status = response['CreateAccountStatus']['State']
            
            if status == 'SUCCEEDED':
                account_id = response['CreateAccountStatus']['AccountId']
                print(f"✅ Account created successfully! Account ID: {account_id}")
                return account_id
            elif status == 'FAILED':
                reason = response['CreateAccountStatus'].get('FailureReason', 'Unknown')
                print(f"ERROR: Account creation failed. Reason: {reason}")
                return None
            else:
                print(f"Status: {status}... (waiting)")
                time.sleep(10)
        except Exception as e:
            print(f"ERROR checking account status: {e}")
            return None

def move_account_to_ou(org_client, account_id, ou_id):
    """
    Move an account to a specific OU.
    
    Args:
        org_client: boto3 Organizations client
        account_id: ID of the account to move
        ou_id: ID of the destination OU
    
    Returns:
        True if successful, False otherwise
    """
    try:
        # First, get the current parent (root)
        roots = org_client.list_roots()
        root_id = roots['Roots'][0]['Id']
        
        print(f"Moving account {account_id} to OU {ou_id}...")
        
        org_client.move_account(
            AccountId=account_id,
            SourceParentId=root_id,
            DestinationParentId=ou_id
        )
        
        print(f"✅ Account moved successfully!")
        return True
    except Exception as e:
        print(f"ERROR moving account: {e}")
        return False

def main():
    parser = argparse.ArgumentParser(
        description='Create a new AWS account in your organization and move it to the specified OU.'
    )
    parser.add_argument('--name', required=True, help='Account name (e.g., cloudguard-workloads-ml-01)')
    parser.add_argument('--email', required=True, help='Root user email address')
    parser.add_argument('--ou', required=True, help='Organizational Unit name (e.g., Workloads)')
    parser.add_argument('--region', default='us-east-1', help='AWS region (default: us-east-1)')
    
    args = parser.parse_args()
    
    # Initialize boto3 client
    try:
        org_client = boto3.client('organizations', region_name=args.region)
    except Exception as e:
        print(f"ERROR: Failed to initialize AWS client: {e}")
        print("Make sure your AWS credentials are configured correctly.")
        sys.exit(1)
    
    # Step 1: Get OU ID
    ou_id = get_ou_id(org_client, args.ou)
    if not ou_id:
        sys.exit(1)
    
    print(f"Found OU '{args.ou}' with ID: {ou_id}")
    
    # Step 2: Create account
    request_id = create_account(org_client, args.name, args.email)
    if not request_id:
        sys.exit(1)
    
    # Step 3: Wait for creation to complete
    account_id = wait_for_account_creation(org_client, request_id)
    if not account_id:
        sys.exit(1)
    
    # Step 4: Move account to OU
    if not move_account_to_ou(org_client, account_id, ou_id):
        print("WARNING: Account was created but could not be moved to the OU.")
        print("You can move it manually in the AWS Console.")
        sys.exit(1)
    
    print("\n" + "="*60)
    print("🎉 SUCCESS! Account setup complete.")
    print("="*60)
    print(f"Account Name: {args.name}")
    print(f"Account ID: {account_id}")
    print(f"OU: {args.ou}")
    print(f"Email: {args.email}")
    print("="*60)

if __name__ == '__main__':
    main()
